<?php

namespace App\Http\Controllers\User;

use Illuminate\Http\Request;
use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Services\OrderService;
use App\Models\Voucher;
use App\Models\Order;
use App\Models\User;
use App\Models\Variation;
use Illuminate\Support\Facades\Http;
// --- নতুন সংযোজন শুরু ---
use App\Models\IdempotencyKey;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
// --- নতুন সংযোজন শেষ ---

class OrderController extends Controller
{
    public function index(Request $request, OrderService $orderService)
    {
        $orders = $orderService->getMine($request->query());
        return view('user.orders', compact('orders'));
    }

    public function addOrder(Request $request, OrderService $orderService)
    {
        // --- সম্পূর্ণ ফাংশনটি পরিবর্তন করা হয়েছে ---

        $idempotencyKey = $request->input('idempotency_key');
        $user = auth()->user();

        // ১. কী ভ্যালিড কিনা চেক করুন
        if (empty($idempotencyKey)) {
            return back()->with('error', 'Invalid request. Please try again.');
        }

        // ২. আপনার পুরোনো ভ্যালিডেশনটি এখানে রাখুন
        $validator = validator()->make($request->all(), [
            'variation_id' => 'required|exists:variations,id',
            'quantity' => 'sometimes|required|integer|min:1',
            'payment_method' => 'required|string'
        ]);

        if ($validator->fails()) {
            return back()->with('error', __('Variation ID and Payment Method fields are required.'));
        }

        // ৩. কী আগে ব্যবহৃত হয়েছে কিনা চেক করুন
        $keyExists = IdempotencyKey::where('key', $idempotencyKey)->exists();
        if ($keyExists) {
            return redirect()->route('user.orders')->with('success', 'Your order is already being processed or is complete.');
        }

        try {
            $orderResult = null; // ফলাফল রাখার জন্য ভ্যারিয়েবল

            // ৪. ডেটাবেজ ট্রানজেকশন শুরু করুন
            DB::transaction(function () use ($request, $user, $idempotencyKey, $orderService, &$orderResult) {
                
                // ৫. সর্বপ্রথম কী-টি ডেটাবেজে সেভ করুন
                IdempotencyKey::create([
                    'user_id' => $user->id,
                    'key' => $idempotencyKey,
                ]);

                // ৬. এখন আপনার OrderService-কে কল করুন
                // যদি সার্ভিসের ভেতরে কোনো সমস্যা হয়, তাহলে transaction স্বয়ংক্রিয়ভাবে rollback হয়ে যাবে
                $orderResult = $orderService->addOrder($request);
            });

            // ৭. যদি সবকিছু ঠিক থাকে, তাহলে সার্ভিসের ফলাফল রিটার্ন করুন
            return $orderResult;

        } catch (\Exception $e) {
            // ৮. কোনো সমস্যা হলে লগ করুন এবং ব্যবহারকারীকে বার্তা দেখান
            Log::error('Order processing failed for user ' . $user->id . ': ' . $e->getMessage());
            return back()->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }
    
    public function auto_verify(Request $request)
    {
        
        $orderId = $request->input('orderid');
        $status  = $request->input('status');
        $content = $request->input('content');

        if (!$orderId) {
            return response()->json(['message' => 'Missing order ID'], 400);
        }

        $order = Order::find($orderId);

        if (!$order) {
            return response()->json(['message' => 'Order not found'], 404);
        }

        if ($status === 'success') {
            $order->status = Status::COMPLETED;
            $order->save();

            return response()->json(['message' => 'Order marked as completed']);
        }

        $order->status = 'processing';
        $order->voucher_code= $content;
        $order->save();
        
        $message = "New Order Receive\n\n".
            "Order Confirmation\n\n".
            "Order ID: {$order->id}\n".
            "Package: {$order->variation->title}\n".
            "Amount: {$order->amount}\n".
            "Type: {$order->product->type}\n".
            "Account Info: {$order->account_info}\n".
            "Status: {$order->status}";
                          
        $this->sendNotification($message);

        if ($content==="Invalid Player ID" || $content==="region does not match"){
            $variation = Variation::findOrFail($order->variation_id);
            $variation->stock += $order->quantity;
            $variation->save();
            $v_id = Variation::where('product_id', gs()->unipin_product_id)->where('provider_product_id', $order->variation->provider_product_id)->first();
            $v_id->stock += $order->quantity;
            $v_id->save();
            $order->status = Status::CANCELLED;
            $order->save();
           
            $autoVoucher = Voucher::where('order_id', $order->id)->first();
            $autoVoucher->status = Status::AVAILABLE;
            $autoVoucher->save();

            $user = User::find($order->user_id);
            if ($user) {
                $user->increment('balance', $order->amount);
            }
        }

        return response()->json(['message' => 'Order marked as processing']);
    }

    public function payNow(Request $request, OrderService $orderService)
    {
        $validator = validator()->make($request->all(), [
            'id' => 'required'
        ]);

        if ($validator->fails()) {
            return back()->with('error', __('Order ID field is required.'));
        }

        return $orderService->payNow($request->id);
    }
    
    private function sendNotification($message) {
        $botToken = gs()->telegram_bot_token;
        $chatId = gs()->telegram_chat_id;
        $url = "https://api.telegram.org/bot{$botToken}/sendMessage";

        $response = Http::post($url, [
            'chat_id' => $chatId,
            'text' => $message,
        ]);
        
        return true;
    }
}
